<?php
namespace App\Controller;

use App\Controller\AppController;
use Cake\Log\Log;
use Cake\ORM\TableRegistry;
require_once ROOT . DS . "src" . DS . "Controller" .  DS . "Component" . DS . "simplexlsx.class.php";

/**
 * SystemMedications Controller
 *
 * @property \App\Model\Table\SystemMedicationsTable $SystemMedications
 */
class SystemMedicationsController extends AppController
{
    public function initialize()
    {
        parent::initialize();
        $this->loadComponent('Auth', [
           'authorize' => 'Controller',
      ]);
        $this->loadComponent('Permission');

        // View or Controller
        $this->set('title', 'Medicamentos/Procedimentos');

        $session = $this->request->session();

        $pacient_id = $session->read('selected-pacient');
        $this->set('pacient_id', $pacient_id);

        $unit_id = $session->read('selected-unit');
        $this->set('unit_id', $unit_id);

        $hospitalization_id = $session->read('selected-hospitalization');
        $this->set('hospitalization_id', $hospitalization_id);

        $current_state = $this->request->session()->read('current-state');
        $this->set('current_state', $current_state);
    }

    public function isAuthorized($user)
    {
        $action = $this->request->params['action'];
        $privilege = $this->request->session()->read('selected-unit-privilege');
        return $this->Permission->hasPermission($privilege, 'SystemMedications', $action);
    }

    /**
     * Index method
     *
     * @return \Cake\Network\Response|null
     */
    public function index()
    {
        $this->set('title', 'Itens');
        $session               = $this->request->session();
        $hospitalizationId     = $session->read('selected-hospitalization');
        $unitId                = $session->read('selected-unit');
        $screening_id          = $session->read('screening_id');
        $current_state         = $session->read('current-state');

        $query = $this->SystemMedications->find('all')
         ->where([
           'is_active' => 1,
           'unit_id'   => $unitId
         ]);

        $searchParam = (!empty($this->request->query('q'))) ? $this->request->query('q') : '';

        if ($searchParam != '') {
            $query = $query
            ->where(['name LIKE' => "%".$searchParam."%"])
            ->orWhere(['category LIKE' => "%".$searchParam."%"])
            ->orWhere(['channel LIKE' => "%".$searchParam."%"])
            ->orWhere(['surveillance_signaling LIKE' => "%".$searchParam."%"]);
        }

        $medications = $this->paginate($query);
        $printParam = $searchParam;
        $this->set(compact('medications', 'hospitalizationId', 'screening_id', 'current_state', 'printParam'));
        $this->set('_serialize', ['medications']);
    }

    /**
     * View method
     *
     * @param string|null $id System Medication id.
     * @return \Cake\Network\Response|null
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $systemMedication = $this->SystemMedications->get($id, [
            'contain' => ['Professionals']
        ]);

        $this->set('systemMedication', $systemMedication);
        $this->set('_serialize', ['systemMedication']);
    }

    /**
     * Add method
     *
     * @return \Cake\Network\Response|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $systemMedication = $this->SystemMedications->newEntity();


        $session                = $this->request->session();
        $hospitalization_id     = $session->read('selected-hospitalization');
        $unitId                 = $session->read('selected-unit');
        $sectorsTable           = TableRegistry::get('Sectors');
        $professionalsTable     = TableRegistry::get('Professionals');
        $specialtiesTable       = TableRegistry::get('Specialties');

        $sectors = $sectorsTable->find('list', [
            'keyField' => 'id',
            'valueField' => 'name'
         ])
         ->where(["Sectors.unit_id" => $unitId]);

        $professionals = $professionalsTable->find('list', [
             'keyField' => 'id',
             'valueField' => 'user.name'
          ])
          ->contain(['Users', 'Sectors'])
          ->where(["Professionals.unit_id" => $unitId]);

        $specialties = $specialtiesTable->find('list', [
              'keyField' => 'id',
              'valueField' => 'name'
           ]);

        $associated = [
           'RestrictedSectors',
           'RestrictedSpecialties',
           'RestrictedProfessionals',
           'Professionals'
         ];

        if ($this->request->is('post')) {

            $systemMedication = $this->SystemMedications->patchEntity($systemMedication, $this->request->data, ["associated" => $associated]);
            if ($this->SystemMedications->save($systemMedication)) {
                $this->Flash->success(__('A medicação foi cadastrada com sucesso.'));
                return $this->redirect(['action' => 'index']);
            } else {
                $this->Flash->error(__('A medicação não pêde ser cadastrada, por favor tente novamente.'));
            }
        }

        $this->set(compact('systemMedication', 'sectors', 'hospitalization_id', 'specialties', 'professionals', 'unitId'));
        $this->set('_serialize', ['systemMedication']);
    }

    /**
     * Edit method
     *
     * @param string|null $id System Medication id.
     * @return \Cake\Network\Response|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Network\Exception\NotFoundException When record not found.
     */
    public function edit($id = null)
    {
        $associated = [
          'RestrictedSectors',
          'RestrictedSpecialties',
          'RestrictedProfessionals',
          'Professionals'
        ];

        $systemMedication = $this->SystemMedications->get($id, [
            'contain' => $associated
        ]);


        $session                = $this->request->session();
        $hospitalizationId     = $session->read('selected-hospitalization');
        $unitId                 = $session->read('selected-unit');
        $current_state          = $session->read('current-state');
        $sectorsTable           = TableRegistry::get('Sectors');
        $professionalsTable     = TableRegistry::get('Professionals');
        $specialtiesTable       = TableRegistry::get('Specialties');

        $sectors = $sectorsTable->find('list', [
            'keyField' => 'id',
            'valueField' => 'name'
         ])
         ->where(["Sectors.unit_id" => $unitId]);

        $professionals = $professionalsTable->find('list', [
             'keyField' => 'id',
             'valueField' => 'user.name'
          ])
          ->contain(['Users', 'Sectors'])
          ->where(["Professionals.unit_id" => $unitId]);

        $specialties = $specialtiesTable->find('list', [
              'keyField' => 'id',
              'valueField' => 'name'
           ]);


        if ($this->request->is(['patch', 'post', 'put'])) {
            $systemMedication = $this->SystemMedications->patchEntity($systemMedication, $this->request->data, ["associated" => $associated]);
            if ($this->SystemMedications->save($systemMedication)) {
                $this->Flash->success(__('A medicação foi cadastrada com sucesso.'));
                return $this->redirect(['action' => 'index']);
            } else {
                $this->Flash->error(__('A medicação não pêde ser cadastrada, por favor tente novamente.'));
            }
        }

        $this->set(compact('systemMedication', 'sectors', 'hospitalizationId', 'specialties', 'current_state', 'professionals',  'unitId'));
        $this->set('_serialize', ['systemMedication']);
    }


    public function printItems()
    {
        $is_print_view = true;

        $session             = $this->request->session();
        $hospitalization_id  = $session->read('selected-hospitalization');
        $unitId              = $session->read('selected-unit');
        $unitsTable          = TableRegistry::get('Units');

        $unit = $unitsTable->get($unitId);

        $query = $this->SystemMedications->find('all')
         ->where([
           'is_active' => 1,
           'unit_id'   => $unitId
         ]);

        $searchParam = (!empty($this->request->query('q'))) ? $this->request->query('q') : '';
        if ($searchParam != '') {
            $query = $query
               ->where(['name LIKE' => "%".$searchParam."%"])
               ->orWhere(['category LIKE' => "%".$searchParam."%"])
               ->orWhere(['channel LIKE' => "%".$searchParam."%"])
               ->orWhere(['surveillance_signaling LIKE' => "%".$searchParam."%"]);
        }

        $medications = $this->paginate($query);

        $this->set(compact('is_print_view', 'medications', 'hospitalizationId','unit'));
        $this->set('_serialize', ['medications']);
    }

    public function importItens()
    {
        $session                = $this->request->session();
        $professionalsTable     = TableRegistry::get('Professionals');

        $unitId             = $session->read('selected-unit');
        $currentUserId      = $session->read('Auth.User.id');

        $professional = $professionalsTable->find('all')
        ->where(['user_id' => $unitId])
        ->first();

        if(!$professional) {
          $this->Flash->error(__('É necessário estar cadastrado como profissional para realizar esta ação.'));
          return $this->redirect($this->referer());
        }

        define("LNBR", PHP_EOL);

        $this->render(false);

        // creating an array with the possible model associations
        $associated = [
           'Professionals'
         ];

        if ($this->request->is('post')) {

            //getting the file from post
            $data     = $this->request->data['sheet'];
            $file     = $data['tmp_name'];
            $filename = $data['name'];
            $allowed  = array('xlsx');
            $errosOnSave=null;

            // verifying file extension
            if (!in_array(substr(strrchr($filename, '.'), 1), $allowed)) {
                $this->Flash->error(__('O tipo de arquivo importado não é compatável, por favor utilize a extensão ".xlsx".'));
                return $this->redirect($this->referer());
            } elseif (empty($current_professional)) {
                $this->Flash->error(__('Você não tem permissão para esse tipo de ação.'));
                return $this->redirect($this->referer());
            }

            $sheetRows = [];
            if ( $xlsx = \SimpleXLSX::parse($file) ) {
            	$sheetRows = $xlsx->rows();
            }

            //initializing medications array and iterator
            $medications = [];
            $i=0;

            // cycle through the rows in the csv file
            foreach ($sheetRows as $key => $row) {

                // checking if row is empty and the number of columns is correct
                if ($row == '' || empty($row) || $row == null || count($row) != 15) {
                    break;
                }

                // checking if name is empty
                if ($row[0]=='' || empty($row[0])) {
                    break;
                }

                $medications[$i] = [
                    "professional_id" => $professional->id,
                    'name' => $row[0],
                    'category' =>$row[1],
                    'presentation' =>$row[2],
                    'quantitity' => (is_numeric($row[3])) ? $row[3] : 0,
                    'channel' =>$row[4],
                    'frequency' => $row[5],
                    'controlled_drug' => (strtolower(($row[6])) == "sim") ? true : false,
                    'is_antibiotic' => (strtolower(($row[7])) == "sim") ? true : false,
                    'justificativa_antibiotico' => $row[8],
                    'is_surveillance_signed' => (strtolower(($row[9])) == "sim") ? true : false,
                    'surveillance_signaling' => $row[10],
                    'is_restricted_by_sectors' => (strtolower(($row[11])) == "sim") ? true : false,
                    'is_restricted_by_users' => (strtolower(($row[12])) == "sim") ? true : false,
                    'unitary_price' => (is_numeric($row[13])) ? $row[13] : 0,
                    'accepts_devolution' => (strtolower(($row[14])) == "sim") ? true : false,
                    'unit_id' => $unitId
                ];

                $i++;
            }


            // checking my beers array and starting to save each entity
            if (!empty($medications)) {

                // elimininado o header de titulos da planilha
                unset($medications[0]);

                $medicationsToSave = count($medications);
                $success = 0;
                $errosOnSave['medications_names']  = [];
                $errosOnSave['modelErrors'] = [];

                foreach ($medications as $key => $medication) {
                    $exists = $this->SystemMedications->exists(['name'=> $medication['name']]);

                    if (!$exists) {
                        $medicationObject = $this->SystemMedications->newEntity();
                        $medicationObject = $this->SystemMedications->patchEntity($medicationObject, $medication, ['associated'=>$associated]);

                        if ($this->SystemMedications->save($medicationObject)) {
                            $success++;
                        } else {
                            array_push($errosOnSave['modelErrors'], $medicationObject->errors());
                            Log::write('debug', $medicationObject->name);
                            Log::write('debug', $medicationObject->errors());
                        }
                    } else {
                        array_push($errosOnSave['medications_names'], $medication['name']);
                    }
                }

                // erros

                if (!empty($errosOnSave['medications_names'])) {
                    $this->Flash->error(__('Alguns registros possuiam nomes que já existiam e não foram importados.'));
                }

                if (!empty($errosOnSave['modelErrors'])) {
                    $this->Flash->error(__('Alguns registros não foram importados. Verifique o preenchimento dos campos e tente novamente.'));
                }

                // sucesso
                if ($success == $medicationsToSave) {
                    $this->Flash->success(__('A importação foi feita com sucesso.'));
                }
            } else {
                $this->Flash->error(__('O arquivo de importação enviado estava vazio ou não foi preenchido corretamente.'));
            }

            return $this->redirect(['action' => 'index']);
      }
    }


    public function deactivate($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);

        $systemMedication = $this->SystemMedications->get($id,[
          'contain' => 'PrescriptionMedications'
        ]);

        $systemMedication->is_active = false;




        if ($this->SystemMedications->save($systemMedication)) {

            foreach ($systemMedication['prescription_medications'] as $key => $prescriptionMedication) {
              $prescriptionMedication->is_active = false;
              $this->SystemMedications->PrescriptionMedications->save($prescriptionMedication);
            }

            $this->Flash->success(__('Desabilitada com sucesso.'));
            
        } else {
            $this->Flash->error(__('Não foi possível desabilitar, por favor tente novamente.'));
        }
        return $this->redirect(['action' => 'index']);
    }

    /**
     * Delete method
     *
     * @param string|null $id System Medication id.
     * @return \Cake\Network\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $systemMedication = $this->SystemMedications->get($id);
        if ($this->SystemMedications->delete($systemMedication)) {
            $this->Flash->success(__('The system medication has been deleted.'));
        } else {
            $this->Flash->error(__('The system medication could not be deleted. Please, try again.'));
        }
        return $this->redirect(['action' => 'index']);
    }
}
